namespace java com.ghca.kms.thrift.api
//签名算法
enum Algorithm {
//RSA加密算法
    RSAES_OAEP_SHA_256 = 1
    RSAES_OAEP_SHA_1 = 2
//    SM2加密
    SM2PKE = 3
//    RSA签名
    RSA_PASS_SHA_256 = 4
    RSA_PKCS1_SHA_256 = 5
//    SM2签名
    SM2DSA = 6
}

//密钥的类型
enum KeySpec {
    RSA_1024 = 1,
    RSA_2048 = 2,
    EC_SM2 = 5,
    AES_256 = 6,
    AES_128 = 7,
    SM4 = 8,
    RSA_3072 = 3,
    RSA_4096 =4
}
// 密钥的用途
enum KeyUsage {
    ENCRYPT_DECRYPT = 1, //加密和解密
    SIGN_VERIFY = 2 //产生和验证数字签名。
}
/*
 * 密钥材料来源
 * 请注意区分大小写。当KeySpec为非对称密钥类型时禁止选择EXTERNAL。如果选择EXTERNAL，您需要导入密钥材料。
 */
enum Origin {
    KMS = 1,
    EXTERNAL = 2
}
/*
   密钥的保护级别
   请注意区分大小写。
   当取值为HSM时，如果Origin参数为KMS，则会在托管密码机中生成密钥，用于执行密码运算；如果Origin参数为EXTERNAL，您可以将外部密钥导入到托管密码机中，用于执行密码运算
 */
enum ProtectionLevel {
    SOFTWARE = 1,
    HSM = 2
}

// 对称加密模式
enum EncMode {
    ECB = 1
    CBC = 2
    OFB = 3
    CFB = 4
    GCM = 5
}

// 补位方式
enum Padding {
    NoPadding = 1
    PKCS5Padding = 2
    PKCS7Padding = 3
}

// 对称加密参数
struct SymParam {
    1: EncMode mode                             // 对称加密模式，如ECB、CBC等，参考EncMode定义
    2: binary  iv                               // 初始化向量， 除ECB外其他都需要
    3: Padding padding =  Padding.PKCS7Padding    // padding 模式，NoPadding时要保证数据长度满足加密算法
}

enum SignAlgorithm {
    ED25519 = 1
    ED25519PH = 2
    ED25519CTX = 3
    RSA_PKCS1_5 = 4
    RSA_SSA_PSS = 5
    SM2 = 6
    ECDSA = 7
}

enum DigestAlgorithm {
    SHA1 = 1
    SHA256 = 2
    SHA512 = 3
    SHA384 = 4
    SHA224 = 5
    NO_PREHASH = 6
    SM3 = 7
}


// KMS服务端异常
exception KmsException {
    1: required string  err        // 错误信息，服务方错误时必须填写
    2: optional i32     errCode    // 错误码,可选
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
//                                    密码运算请求/返回                                               //
//////////////////////////////////////////////////////////////////////////////////////////////////////

//5.2.1 encrypt
struct EncryptRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required string plaintext             // 待加密明文（必须经过Base64编码）
    3:map<string,string> encryptionContext  // key/value的JSON字符串。如果指定了该参数，则在调用Decrypt时需要提供同样的参数.
    4:required string accessToken           // 登录后返回的token
}
struct EncryptResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 用于加密明文的密钥版本标志符。是指定CMK的主版本
    3:string ciphertextBlob                 // 数据被指定CMK的主版本加密后的密文。
    4:string requestId       	            // 当前请求的标志符。
}

//5.2.2 generateDataKey
struct GenerateDataKeyRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:KeySpec keySpec = KeySpec.SM4         // 指定生成的数据密钥的长度，取值：
    3:i32 numberOfBytes                     // 指定生成的数据密钥的长度。取值：1~1024
    4:map<string,string> encryptionContext  // key/value对的JSON字符串。如果指定了该参数，则在调用Decrypt时需要提供同样的参数
    5:required string accessToken           // 登录后返回的token
}
struct GenerateDataKeyResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 用于加密明文的密钥版本标志符。是指定CMK的主版本
    3:string ciphertextBlob                 // 数据密钥被指定CMK的主版本加密后的密文。
    4:string plaintext                      // 数据密钥的明文经过Base64编码的后的值.
    5:string requestId       		        // 当前请求的标志符。
    6:string dataKeyId
    7:string kcv
}

//5.2.3 generateDataKeyWithoutPlaintext
struct GenerateDataKeyWithoutPlaintextRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:KeySpec keySpec = KeySpec.SM4         // 指定生成的数据密钥的长度，取值：
    3:i32 numberOfBytes                     // 指定生成的数据密钥的长度。取值：1~1024
    4:map<string,string> encryptionContext  // key/value对的JSON字符串。如果指定了该参数，则在调用Decrypt时需要提供同样的参数
    5:required string accessToken           // 登录后返回的token
}
struct GenerateDataKeyWithoutPlaintextResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 用于加密明文的密钥版本标志符。是指定CMK的主版本
    3:string ciphertextBlob                 // 数据密钥被指定CMK的主版本加密后的密文。
    4:string requestId       		        // 当前请求的标志符。
    5:string dataKeyId
}

//5.2.4 decrypt
struct DecryptRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required string ciphertextBlob        // 待解密的密文
    3:map<string,string> encryptionContext  // key/value对的JSON字符串。如果指定了该参数，则在调用Decrypt时需要提供同样的参数
    4:required string accessToken           // 登录后返回的token
}
struct DecryptResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string plaintext                      // 解密后的明文。
    4:string requestId       		        // 当前请求的标志符。
}

//5.2.5 asymmetricSign
struct AsymmetricSignRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:required Algorithm algorithm          // 签名算法,如SM2_SM3
    4:required string digest                // 使用Algorithm中对应的哈希算法，对原始消息生成的摘要，使用Base64编码。
    5:required string accessToken           // 登录后返回的token
}
struct AsymmetricSignResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string value                          // 计算出来的签名，使用Base64编码。
    4:string requestId       		        // 当前请求的标志符。
}

struct AsymmetricSignRequest2 {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required binary digest                // 使用Algorithm中对应的哈希算法，对原始消息生成的摘要。
    3:required string accessToken           // 登录后返回的token
    4:SignAlgorithm signAlgorithm
    5:binary context
    6:DigestAlgorithm digestAlgorithm       //摘要算法
    7:string keyVersionId                   // 密钥版本的全局唯一标识符。
}

struct AsymmetricSignResponse2 {
    1:string keyId                          // CMK的全局唯一标识符。
    2:binary value                          // 计算出来的签名
    3:string requestId       		        // 当前请求的标志符。
    4:string keyVersionId                   // 密钥版本的全局唯一标识符。
}

//5.2.6 asymmetricVerify
struct AsymmetricVerifyRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:string keyVersionId          // 密钥版本的全局唯一标识符。
    3:required Algorithm algorithm             // 签名算法,如SM2_SM3
    4:required string digest                // 使用Algorithm中对应的哈希算法，对原始消息生成的摘要，使用Base64编码。
    5:required string value                 // 待验证的签名值，使用Base64编码.
    6:required string accessToken           // 登录后返回的token
}
struct AsymmetricVerifyResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string value                          // 签名是否通过，true/false
    4:string requestId       		        // 当前请求的标志符。
}

struct AsymmetricVerifyRequest2 {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required binary digest                // 使用Algorithm中对应的哈希算法，对原始消息生成的摘要
    3:required binary value                 // 待验证的签名值
    4:required string accessToken           // 登录后返回的token
    5:SignAlgorithm signAlgorithm
    6:binary context
    7:DigestAlgorithm digestAlgorithm       //摘要算法
    8:string keyVersionId                   // 密钥版本的全局唯一标识符。
}


struct AsymmetricVerifyResponse2 {
    1:string keyId                          // CMK的全局唯一标识符。
    2:bool value                            // 签名是否通过，true/false
    3:string requestId       		        // 当前请求的标志符。
    4:string keyVersionId                   // 密钥版本的全局唯一标识符
}

//5.2.7 asymmetricEncrypt
struct AsymmetricEncryptRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:string keyVersionId          // 密钥版本的全局唯一标识符。
    3:required Algorithm algorithm          // 签名算法,如SM2_SM3
    4:required string plaintext             // 要加密的明文，使用Base64编码
    5:required string accessToken           // 登录后返回的token
}
struct AsymmetricEncryptResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string ciphertextBlob                 // 加密后的密文，使用Base64编码
    4:string requestId       		        // 当前请求的标志符。
}

struct AsymmetricEncryptRequest2 {
     1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
     2:required string plaintext             // 要加密的明文，使用Base64编码
     3:required string accessToken           // 登录后返回的token
 }
 struct AsymmetricEncryptResponse2 {
     1:string keyId                          // CMK的全局唯一标识符。
     2:binary ciphertextBlob                 // 加密后的密文
     3:string requestId       		        // 当前请求的标志符。
 }

//5.2.8 asymmetricDecrypt
struct AsymmetricDecryptRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:string keyVersionId          // 密钥版本的全局唯一标识符。
    3:required Algorithm algorithm          // 签名算法,如SM2_SM3
    4:required string ciphertextBlob        // 待解密的密文，使用Base64编码
    5:required string accessToken           // 登录后返回的token
}
struct AsymmetricDecryptResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string plaintext                      // 解密后的明文，使用Base64编码
    4:string requestId       		        // 当前请求的标志符。
}
struct AsymmetricDecryptRequest2 {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    4:required binary ciphertextBlob        // 待解密的密文
    5:required string accessToken           // 登录后返回的token
}
struct AsymmetricDecryptResponse2 {
    1:string keyId                          // CMK的全局唯一标识符。
    3:binary plaintext                      // 解密后的明文，使用Base64编码
    4:string requestId       		        // 当前请求的标志符。
}

//5.2.8 getPublicKey
struct GetPublicKeyRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:string keyVersionId          // 密钥版本的全局唯一标识符。
    3:required string accessToken           // 登录后返回的token
}
struct GetPublicKeyResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:string keyVersionId                   // 密钥版本的全局唯一标识符。
    3:string publicKey                      // 返回的公钥
    4:string requestId       		        // 当前请求的标志符。
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
//                                  用户CMK别名管理                                                   //
//////////////////////////////////////////////////////////////////////////////////////////////////////
/**
* 创建别名接口参数
**/
struct CreateAliasRequest{
    //要操作的别名,长度为1~255个字符,必须包含前缀alias
    1:string aliasName;
    //CMK的全局唯一标识符。
    2:string keyId;
    3:required string accessToken;
}
/**
* 创建别名接口返回
**/
struct CreateAliasResponse{
    //本次请求的ID
    1:string requestId;
}

/**
* 更新别名接口参数
**/
struct UpdateAliasRequest{
    //要操作的别名,长度为1~255个字符,必须包含前缀alias
    1:string aliasName;
    //CMK的全局唯一标识符。
    2:string keyId;
    3:required string accessToken;
}
/**
* 更新别名接口返回
**/
struct UpdateAliasResponse{
    //本次请求的ID
    1:string requestId
}
/**
* 删除别名接口参数
**/
struct DeleteAliasRequest{
    //要操作的别名,长度为1~255个字符,必须包含前缀alias
    1:string aliasName;
    2:required string accessToken;
}
/**
* 删除别名接口返回
**/
struct DeleteAliasResponse{
    //本次请求的ID
    1:string requestId
}
/**
* 列出CMK别名请求参数
**/
struct ListAliasRequest{
    //当前页数，如果未传入，则默认为第1页
    1:i32 pageNumber;
    //每页显示的条数，如果传入，则默认为10条
    2:i32 pageSize;
    3:required string accessToken;
    }
/**
* 列出CMK别名返回结果
**/
struct ListAliasResponse{
    // 返回的CMK总数。
    1:i32 totalCount;
    // 当前页数。
    2:i32 pageNumber;
    // 每页的返回结果个数。
    3:i32 pageSize
    //本次请求的ID。
    4:string requestId
    //用户别名。
    5:list<Alias> aliases
}
/**
* ListAliasResponse内别名
**/
struct Alias{
    1:string keyId;
    //别名的唯一标识符。
    2:string aliasName;
    //别名的ARN。
    3:string aliasArn;
}
/**
* 查询特定cmk
**/
struct ListAliasesByKeyIdRequest{
    //CMK的全局唯一标识符。
    1:string keyId;
    // 当前页数，如果未传入，则默认为第1页
    2:i32 pageNumber;
    // 每页显示的条数，如果传入，则默认为10条
    3:i32 pageSize;
    4:required string accessToken;
}
//////////////////////////////////////////////////////////////////////////////////////////////////////
//                                  用户CMK版本管理                                                   //
//////////////////////////////////////////////////////////////////////////////////////////////////////
/**
* 用于查询指定的密钥版本的元数据信息
**/
struct DescribeKeyVersionRequest{
// CMKID
1:string keyId;
// CMK版本ID
2:string keyVersionId;
3:required string accessToken;
}
/**
* 用于返回指定的密钥版本的元数据信息
**/
struct DescribeKeyVersionResponse{
// CMKID
1:string requestId;
// CMK版本
2:KeyVersion keyVersion;
}
/**
* CMK 版本信息
**/
struct KeyVersion{
// CMKID
1:string keyId;
// CMK版本ID
2:string keyVersionId;
// 创建密钥版本时间
3:string creationDate;
}
/**
* 用于查询主密钥的所有密钥版本
**/
struct ListKeyVersionsRequest{
// CMKID
1:string keyId;
// 每页大小
2:i32 pageSize;
// 当前页
3:i32 pageNumber;
4:required string accessToken;
}
/**
* 主密钥的所有密钥版本
**/
struct ListKeyVersionsResponse{
// requestId
1:string requestId;
// 每页大小
2:i32 pageSize;
// 当前页
3:i32 pageNumber;
// 返回的密钥版本总数
4:i32 totalCount;
5:list<KeyVersion> keyVersions;
}
/**
* 更新密钥轮转策略请求
**/
struct UpdateRotationPolicyRequest{
// CMKID
1:string keyId;
// 是否开启自动密钥轮转
2:bool enableAutomaticRotation;
// 自动轮转的时间周期。格式为integer[unit]，其中integer表示时间长度，unit表示时间单位。合法的unit单位为：d（天）、h（小时）、m（分钟）、s（秒）。7d或者604800s均表示7天的周期。取值：7~730天。
//说明 当EnableAutomaticRotation参数为true时，必须设置此参数；反之，将忽略此参数
3:string rotationInterval;
4:required string accessToken;
}
/**
* 更新密钥轮转策略
**/
struct UpdateRotationPolicyResponse{
// requestId
1:string requestId;
}
/**
* 为主密钥创建一个新的密钥版本
**/
struct RotateKeyRequest{
// CMKID
1:string keyId;
2:required string accessToken;

}
/**
* 为主密钥创建一个新的密钥版本后返回结果
**/
struct RotateKeyResponse{
// requestId
1:string requestId;
// CMK版本
2:KeyVersion keyVersion
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
//                                    用户主密钥管理请求/返回                                               //
//////////////////////////////////////////////////////////////////////////////////////////////////////
struct KeyMetadata {
    1:string arn, // 资源名称
    2:string automaticRotation, //是否开启自动密钥轮转: enabled：自动轮转处于开启状态。 disabled：自动轮转处于未开启状态。suspended：自动轮转被暂停执行
    3:string creationDate, //创建主密钥的日期和时间（UTC）
    4:string creator, //主密钥创建者
    5:string deleteDate, // 主密钥的删除时间。
    6:string description, // 主密钥的描述。
    7:string keyId, // 主密钥的全局唯一标识符
    8:string keySpec, // 主密钥的类型。
    9:string keyState, // 主密钥的状态。
    10:string keyUsage, // 主密钥的用途
    11:string lastRotationDate, // 最近一次轮转的时间。 如果是新创建密钥，则为初始密钥版本生成时间。
    12:string materialExpireTime, //密钥材料的过期时间。当该值为空时，表示密钥材料不会过期。
    13:string nextRotationDate, //下一次轮转的时间。
    14:string origin, //主密钥的密钥材料来源。
    15:string protectionLevel, //密钥的保护级别。
    16:string rotationInterval, // 密钥自动轮转的周期（秒数）。格式为整数值后加上字符s。例如，7天的轮转周期为604800s。只有当automaticRotation参数值为Enabled或Suspended时，返回该值。
    17:optional string importPublicKey, //当CMK为external时，用来加密导入密钥的公钥.公钥是base64编码
    18:optional string importPublicKeySpec //当CMK为external时，用来加密导入密钥的公钥的类型，此处固定为SM2
}

//5.1.1 CreateKey
struct CreateKeyRequest {
    1: string description, //密钥的描述
    2: KeySpec keySpec, // 密钥的类型，取值：SM4、ESM2
    3: KeyUsage keyUsage, // 密钥的用途
    4: Origin origin, //密钥材料来源
    5: ProtectionLevel protectionLevel, //密钥的保护级别
    6: bool enableAutomaticRotation, // 是否开启自动密钥轮转
    7: string rotationInterval, // 自动轮转的时间周期
    8: string accessToken,
    9: string autoArchiveDate //自动归档日期，格式为yyyy-MM-dd
}

struct CreateKeyResponse {
    1: KeyMetadata keyMetadata, //主密钥（CMK）的metadata
    2: string requestId //请求ID。
}

//5.1.2 importKey
struct ImportKeyRequest {
    1: required string encryptedKeyMaterial, //在createKey后返回的公钥加密并用base64编码后的密钥材料
    2: required string keyId,//待导入的主密钥ID
    3: required i64 keyMaterialExpireUnix, // 密钥材料过期时间。
    4: string accessToken
}

struct ImportKeyResponse {
    1: string requestId //请求ID。
}

//5.1.4 enableKey
struct EnableKeyRequest {
    1: required string keyId, //待启用的主密钥ID
    2: string accessToken
}

struct EnableKeyResponse {
    1: string requestId //请求ID。
}

//5.1.5 disableKey
struct DisableKeyRequest {
    1: required string keyId, //待禁用的主密钥ID
    2: string accessToken
}

struct DisableKeyResponse {
    1: string requestId //请求ID。
}

//5.1.6 deleteKey
struct DeleteKeyRequest {
    1: required string keyId, //CMK 全局唯一标识符。
    2: string accessToken
}

struct DeleteKeyResponse {
    1: string requestId //请求ID。
}

//5.1.9 describeKey
struct DescribeKeyRequest {
    1: required string keyId, //待操作的主密钥ID。
    2: string accessToken
}

struct DescribeKeyResponse {
   1: KeyMetadata keyMetadata, //主密钥（CMK）的metadata。
   2: string requestId //请求ID。
}

//5.1.10 listKeys
struct Key {
   1: string keyId, //主密钥的全局唯一标识符。
   2: string keyArn, //主密钥的ARN。
   3: i32 keyState //密钥状态
}

struct ListKeysRequest {
   1: i32 pageNumber = 1 //当前页数
   2: i32 pageSize = 10 //每页返回值的个数
   3: string accessToken
}

struct ListKeysResponse {
   1: i32 totalCount, //主密钥的总数
   2: i32 pageNumber, //当前页数
   3: i32 pageSize, // 每页返回值的个数
   4: string requestId, //请求ID。
   5: list<Key> keys //主密钥
}

//5.1.11 updateKeyDescription
struct UpdateKeyDescriptionRequest {
    1: required string keyId, //主密钥（CMK）的全局唯一标识符
    2: required string description //CMK的描述性信息，通常用于描述CMK的用途
    3: string accessToken
}

struct UpdateKeyDescriptionResponse {
   1: string requestId //请求ID。
}

//5.4.1 登录认证接口
struct TokenRequest{
//用户对应的唯一访问key
   1: required string accesskey
//   随机数
   2: required string random
//   生成数据的时间
   3: required string dateTime
//访问key对应的密钥
   4: required string secretKey
}
struct TokenResponse{
   1: string accessToken //返回的token
   2: string expireTime //过期时间，格式 yyyy-MM-dd HH24:mi:ss
   3: optional string hasKcv // 1-校验kcv 0-不校验
}

struct CreateTenantKeyRequest{
//请求的token
   1: required string accessToken
//需要创建的租户
   2: required string tenantId
}

struct CreateTenantKeyResponse{
   1: string requestId
}

//加密secretKey请求
struct EncryptSecretKeyRequest{
   1: required string tenantId
   2: required string secretKey
   3: required string accessToken
}

struct EncryptSecretKeyResponse{
   1: string requestId
   2: string secretKey
}

//解密secretKey请求
struct DecryptSecretKeyRequest{
   1: required string tenantId
   2: required string secretKey
   3: required string accessToken
}

struct DecryptSecretKeyResponse{
   1: string requestId
   2: string secretKey
}

struct ListDataKeyRequest{
    1: required string accessToken
    2: string page //如果需要翻页就要传这个参数，值是从第一页的返回值nextPage来的
    3: string cmkId
}

struct DataKey{
    1: string cipherKey
    2: string plainKey
    3: string id
    4: optional string kcv
}

struct ListDataKeyResponse{
    1: string requestId
    2: list<DataKey> keys
    3: string nextPage //如果nextPage不为空，表明可能还存在下一页数据。
}

struct HeartBeatRequest {
    1: required string accessKey
    2: required string information
}

struct HeartBeatResponse {
    1: string requestId
    2: string information
}

struct ChallengeRequest {
    1: required string accessKey
}

struct ChallengeResponse {
    1: string challengeCode
}

struct ChallengeTokenRequest {
    1: required string accessKey
    2: required string secret
}

struct ChallengeTokenResponse {
    1: string accessToken //返回的token
    2: string expireTime //过期时间，格式 yyyy-MM-dd HH24:mi:ss
    3: optional string hasKcv // 1-校验kcv 0-不校验
}

struct GetDataKeyRequest {
    1: required string accessToken
    2: required string dataKeyId
}

struct GetDataKeyResponse {
    1: string requestId
    2: string key
    3: optional string kcv
}
struct KcvRequest {
    1:required string accessToken
    2:required string cmkId
    3:required string ciphertext
}
struct KcvResponse {
    1:optional string kcv
}
struct GenerateKeyPairRequest {
    1: KeySpec keySpec             // 密钥类型
    2: required string accessToken // 登录token
}

struct GenerateKeyPairResponse {
    1: string requestId
    2: string publicKey            // Base64格式明文公钥
    3: string plainPrivateKey      // 使用SK加密后的私钥明文（SM4/GCM/NoPadding）
}

struct GenerateDataKeyPairRequest {
    1: required string keyId       // CMK主键或别名
    2: KeySpec keySpec             // 密钥类型
    3: required string accessToken // 登录token
}

struct GenerateDataKeyPairResponse {
    1: string requestId
    2: string publicKey            // Base64格式明文公钥
    3: string plainPrivateKey      // 使用SK加密后的私钥明文（SM4/GCM/NoPadding）
    4: string encryptedPrivateKey  // 使用CMK加密后的私钥明文
    5: string dataKeyId            // dataKey主键
}

struct GetDataKeyPairRequest {
    1: required string accessToken
    2: required string dataKeyId
}

struct GetDataKeyPairResponse {
    1: string requestId
    2: string publicKey
    3: string privateKey // 返回由SK加密的私钥,算法为GCM
}

struct GenerateRandomRequest {
    1: required string accessToken
    2: required i32 byteLength
}

struct GenerateRandomResponse {
    1: string requestId
    2: binary random
}

struct ArchiveRequest {
    1: required string accessToken
    2: required string keyId
    3: required bool archive
}

struct ArchiveResponse {
    1: string requestId
}

struct Pkcs11Attribute{
        1:i32 type
        2:binary value
        3:i32 valueLen
}

struct Pkcs11Object{
        1:i32 id
        2:list<Pkcs11Attribute> templates
        3:i32 count
}

struct CreatePkcs11ObjectRequest{
        1: required string accessToken
        2: list<Pkcs11Attribute> templates
        3: i32 count
}

struct CreatePkcs11ObjectResponse{
        1:i32 objectId
        2:string requestId
}

struct GetPkcs11ObjectRequest{
        1: required string accessToken
        2:i32 objectId
}

struct GetPkcs11ObjectResponse{
        1: list<Pkcs11Attribute> templates
        2: i32 count
        3: string requestId
}

struct ListPkcs11ObjectRequest{
        1: required string accessToken
}

struct ListPkcs11ObjectResponse{
        1: list<Pkcs11Object> objects
        2:i32 count
        3: string requestId
}

struct DeletePkcs11ObjectRequest {
    1:required string accessToken
    2:required i32 objectId
}

struct DeletePkcs11ObjectResponse {
    1: string requestId
}

enum DigestType {
    SHA256 = 1,
    SM3 = 2,
    SHA1 = 3,
    SHA224 = 4,
    SHA384 = 5,
    SHA512 = 6
}

struct SignWithHashedRequest {
    1:required string accessToken
    2:required string keyId
    3:required string digest
    4:required DigestType digestType
}

struct SignWithHashResponse {
    1: string requestId
    2: string value
    3: string keyId
}

struct VerifyWithHashedRequest {
    1:required string accessToken
    2:required string keyId
    3:required string digest
    4:required DigestType digestType
    5:required string sign
}

enum Mode {
    CBC = 1,
    CTR = 2,
    GCM = 3
}

struct EncryptByDataKeyRequest {
    1: required Mode mode
    2: required string plainText
    3: required string keyId
    4: string dataId
    5: required string accessToken
    6: string cipherKey

}

struct EncryptByDataKeyResponse {
    1: required string cipherText
}

struct DecryptByDataKeyRequest {
    1: required Mode mode
    2: required string cipherText
    3: string dataId
    4: required string accessToken
    6: string cipherKey

}

struct DecryptByDataKeyResponse {
    1: required string plainText
}

enum DeriveAlgorithm {
    AES_CMAC = 8384
    TRI_DES_CMAC = 4288
    SHA1 = 2
    SHA224 = 32
    SHA256 = 4
    SHA384 = 16
    SHA512 = 8
}

struct DeriveKeyRequest {
    1: required string accessToken
    2: required string keyId
    3: required DeriveAlgorithm algorithm = DeriveAlgorithm.AES_CMAC
    4: required i32 byteLength
    5: binary salt
}

struct DeriveKeyResponse {
    1: string requestId
    2: binary key // 为防止网络传输泄露数据，该字段使用sk加密，加密算法为SM4/GCM/NoPadding
}

enum SymmetricAlgo {
    SM4 = 1
    AES = 2
}

struct ExportSymmetricKeyRequest {
    1: required string accessToken
    2: required string exportKeyId
    3: required string protectionKeyId
    4: i32 envelopeByteLength=32
    5: SymmetricAlgo envelopeEncAlgo=SymmetricAlgo.AES
    6: EncMode envelopeEncMode = EncMode.CBC
    7: binary  envelopeEncIv
    8:binary envelopeEncAad
}

struct ExportSymmetricKeyResponse {
    1: string requestId
    2: binary envelope
    3: binary exportKey
    4: binary crc
}

struct HmacRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required string plaintext             // 待加密明文（必须经过Base64编码）
    3:DigestAlgorithm digestAlgorithm       // 摘要算法
    4:required string accessToken           // 登录后返回的token
}

struct HmacResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:binary hmactextBlob                   // 数据被指定CMK的主版本hmac后的密文。
    3:string requestId       	            // 当前请求的标志符。
}

struct HmacVerifyRequest {
    1:required string keyId                 // 主密钥（CMK）的全局唯一标识符。该参数也可以被指定为CMK绑定的别名
    2:required string plaintext             // 待加密明文（必须经过Base64编码）
    3:required string hmactext              // hmc结果（必须经过Base64编码）
    4:DigestAlgorithm digestAlgorithm       // 摘要算法
    5:required string accessToken           // 登录后返回的token
}

struct HmacVerifyResponse {
    1:string keyId                          // CMK的全局唯一标识符。
    2:bool result                           // hmac校验结果。
    3:string requestId       	            // 当前请求的标志符。
}

struct PingRequest {
    1: required string ping
}

struct PingResponse {
    1: string pong
}

service KmsApi {
    /**
    * 裸签接口，要求传入数据必须是hash后的值
    */
    SignWithHashResponse signWithHash(SignWithHashedRequest request) throws (1:KmsException ex)

    /**
    * 裸签验证接口
    */
    SignWithHashResponse verifyWithHash(VerifyWithHashedRequest request) throws (1:KmsException ex)

    /**
    * 删除pkcs11对象
    */
    DeletePkcs11ObjectResponse deletePkcs11Object(DeletePkcs11ObjectRequest request) throws (1:KmsException ex)
    /**
    * 创建pkcs11对象，并返回该对象的id
    */
    CreatePkcs11ObjectResponse createPkcs11Object(CreatePkcs11ObjectRequest request) throws (1:KmsException ex)
    /**
    * 通过id获取object信息
    */
    GetPkcs11ObjectResponse getPkcs11Object(GetPkcs11ObjectRequest request) throws (1:KmsException ex)
    /**
    *返回pkcs11对象列表
    */
    ListPkcs11ObjectResponse listPkcs11Object(ListPkcs11ObjectRequest request) throws (1:KmsException ex)

    ArchiveResponse archive(ArchiveRequest request) throws (1:KmsException ex)
    /**
    * 生成指定长度的真随机数
    *    注：长度单位是字节
    */
    GenerateRandomResponse generateRandom(GenerateRandomRequest request) throws (1:KmsException ex)

    /**
     * 生成非对称DataKey
     */
    GenerateKeyPairResponse generateKeyPair(GenerateKeyPairRequest request) throws (1:KmsException ex)

    /**
     * 生成非对称DataKey
     */
    GenerateDataKeyPairResponse generateDataKeyPair(GenerateDataKeyPairRequest request) throws (1:KmsException ex)

    /**
    * 通过ID获取已生成的DataKey
    */
    GetDataKeyPairResponse getDataKeyPair(GetDataKeyPairRequest request) throws (1:KmsException ex)

    /**
    * 通过ID获取已生成的DataKey
    */
    GetDataKeyResponse getDataKey(GetDataKeyRequest request) throws (1:KmsException ex)

    /**
    * 获取挑战码
    */
    ChallengeResponse getChallenge(ChallengeRequest request) throws (1:KmsException ex)
    /**
    * 使用挑战码获取AccessToken
    */
    ChallengeTokenResponse getChallengeToken(ChallengeTokenRequest request) throws (1:KmsException ex)

    /**
     * 心跳检测。传入accessKey以及用sk加密的时间戳字符串。
     * 返回值information格式为TRUE再加上时间戳字符串，并用sk加密
     */
    HeartBeatResponse heartBeat(HeartBeatRequest request) throws (1:KmsException ex)

    /**
    * 获取当前AccessKey对应生成的所有历史DataKey数据。如果nextPage不为空，表明可能还存在下一页数据。
    * 如果cmkId不为空，数据范围会缩小到该cmk创建的范围。
    */
    ListDataKeyResponse listDataKey(ListDataKeyRequest request) throws (1:KmsException ex)
//    string greet(1:string name)

    //////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                      密码运算接口                                                  //
    //////////////////////////////////////////////////////////////////////////////////////////////////////

    /** 5.2.1 encrypt
    * 使用指定用户主密钥加密少量数据。对对称算法，数据量不大于4k；对于非对称算法，数据量不大于该算法一次加密的数据量。
    */
    EncryptResponse encrypt(EncryptRequest request) throws (1:KmsException ex)

    /** 5.2.10 encryptNp
    * 使用指定用户主密钥加密少量数据。对称加密算法：对返回数据不padding，数据量必须是16的整倍数，数据量不大于4k；非对称算法：数据量不大于该算法一次加密的数据量。
    */
    EncryptResponse encryptNp(EncryptRequest request) throws (1:KmsException ex)

	/** 5.2.2 generateDataKey
	* 调用GenerateDataKey接口生成一个随机的数据密钥，您可以用数据密钥进行本地数据的加密。
    */
    GenerateDataKeyResponse generateDataKey(GenerateDataKeyRequest request) throws (1:KmsException ex)

    /** 5.2.11 generateDataKeyNp
	* 调用GenerateDataKeyNp接口生成一个随机的数据密钥，您可以用数据密钥进行本地数据的加密。密钥密文不会padding
    */
    GenerateDataKeyResponse generateDataKeyNp(GenerateDataKeyRequest request) throws (1:KmsException ex)

	/** 5.2.3 generateDataKeyWithoutPlaintext
    * 生成一个随机的数据密钥，您可以用数据密钥进行本地数据的加密。
    */
    GenerateDataKeyWithoutPlaintextResponse generateDataKeyWithoutPlaintext(GenerateDataKeyWithoutPlaintextRequest request) throws (1:KmsException ex)

    /** 5.2.12 generateDataKeyWithoutPlaintextNp
    * 生成一个随机的数据密钥，您可以用数据密钥进行本地数据的加密。密钥密文不会padding
    */
    GenerateDataKeyWithoutPlaintextResponse generateDataKeyWithoutPlaintextNp(GenerateDataKeyWithoutPlaintextRequest request) throws (1:KmsException ex)

	/** 5.2.4 decrypt
	* 解密CiphertextBlob中的密文。
    */
    DecryptResponse decrypt(DecryptRequest request) throws (1:KmsException ex)
	/** 5.2.13 decryptNp
	* 解密CiphertextBlob中的密文。数据必须是不做padding的，长度必须是16的整倍数
    */
    DecryptResponse decryptNp(DecryptRequest request) throws (1:KmsException ex)

    /** 5.2.5 asymmetricSign
    * 使用非对称密钥进行签名。
    */
    AsymmetricSignResponse asymmetricSign(AsymmetricSignRequest request) throws (1:KmsException ex)

    AsymmetricSignResponse2 asymmetricSign2(AsymmetricSignRequest2 request) throws (1:KmsException ex)

	/** 5.2.6 asymmetricVerify
	* 使用非对称密钥进行签名。
    */
    AsymmetricVerifyResponse asymmetricVerify(AsymmetricVerifyRequest request) throws (1:KmsException ex)

    AsymmetricVerifyResponse2 asymmetricVerify2(AsymmetricVerifyRequest2 request) throws (1:KmsException ex)

	/** 5.2.7 asymmetricEncrypt
	* 使用非对称密钥进行加密。
    */
    AsymmetricEncryptResponse asymmetricEncrypt(AsymmetricEncryptRequest request) throws (1:KmsException ex)

    AsymmetricEncryptResponse2 asymmetricEncrypt2(AsymmetricEncryptRequest2 request) throws (1:KmsException ex)

	/** 5.2.8 asymmetricDecrypt
	* 使用非对称密钥进行解密。
    */
    AsymmetricDecryptResponse asymmetricDecrypt(AsymmetricDecryptRequest request) throws (1:KmsException ex)

    AsymmetricDecryptResponse2 asymmetricDecrypt2(AsymmetricDecryptRequest2 request) throws (1:KmsException ex)

	/** 5.2.9 getPublicKey
	* 获取非对称密钥的公钥。用户可以在本地使用公钥进行加密、验签。
    */
    GetPublicKeyResponse getPublicKey(GetPublicKeyRequest request) throws (1:KmsException ex)

    //////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                      用户主密钥管理接口                                                  //
    //////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
    *5.1.2 importKey
     调用ImportKey接口导入密钥材料
     说明
     对密钥类型为SM4的CMK，密钥材料必须为256位
     您可以为密钥材料设置过期时间，也可以设置其永不过期。
     您可以随时为指定的CMK重新导入密钥材料，并重新指定过期时间，但必须导入相同的密钥材料。
     导入的密钥材料过期或者被删除后，指定的CMK将无法使用，需要再次导入相同的密钥材料才可正常使用。
     同样的密钥材料可导入不同的CMK中，但使用其中一个CMK加密的数据或生成的数据密钥（Data Key）无法使用另一个CMK解密。
    **/
    CreateKeyResponse createKey(1: CreateKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.2 importKey
     调用ImportKey接口导入密钥材料
     说明
     对密钥类型为SM4的CMK，密钥材料必须为256位
     您可以为密钥材料设置过期时间，也可以设置其永不过期。
     您可以随时为指定的CMK重新导入密钥材料，并重新指定过期时间，但必须导入相同的密钥材料。
     导入的密钥材料过期或者被删除后，指定的CMK将无法使用，需要再次导入相同的密钥材料才可正常使用。
     同样的密钥材料可导入不同的CMK中，但使用其中一个CMK加密的数据或生成的数据密钥（Data Key）无法使用另一个CMK解密。
    **/
    ImportKeyResponse importKey(1: ImportKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.4 enableKey
     将一个指定的 CMK 标记为启用状态，可以使用它进行加解密
    **/
    EnableKeyResponse enableKey(1: EnableKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.5 disableKey
     将一个指定的主密钥（CMK）标记为禁用状态。处于禁用状态的 CMK 无法用于加密、解密操作；恢复至启用状态之前，原来使用该 CMK 加密的密文也无法解密。
    **/
    DisableKeyResponse disableKey(1: DisableKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.6 deleteKey
     申请删除一个指定的主密钥（CMK)。
     主密钥一旦被删除无法恢复，使用该主密钥加密的内容与使用该主密钥产生的数据密钥（Datakey）均无法再被解密。因此，我们不提供直接删除主密钥的功能，而是采用申请删除的方式。并且我们建议您尽可能选择密钥禁用DisableKey。
    **/
    DeleteKeyResponse deleteKey(1: DeleteKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.9 describeKey
     返回指定主密钥（CMK）的相关信息。
    **/
    DescribeKeyResponse describeKey(1: DescribeKeyRequest request) throws (1:KmsException ex)

    /**
    *5.1.10 listKeys
     调用ListKeys查询调用者在调用区域的所有主密钥ID
    **/
    ListKeysResponse listKeys(1: ListKeysRequest request) throws (1:KmsException ex)

    /**
    *5.1.11 updateKeyDescription
     更新主密钥的描述信息。
     将主密钥的描述信息（DescribeKey中的Description属性）替换为用户传入的值。使用此API可以对密钥的描述信息进行添加、变更、删除操作
    **/
    UpdateKeyDescriptionResponse updateKeyDescription(1: UpdateKeyDescriptionRequest request) throws (1:KmsException ex)


    //////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                  用户CMK别名管理                                                   //
    //////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
    * 给主密钥（CMK）创建一个别名。
    **/
    CreateAliasResponse createAlias(1:CreateAliasRequest createAliasRequest)throws (1:KmsException ex)

    /**
    * 更新已存在的别名所代表的主密钥（CMK）
    **/
    UpdateAliasResponse updateAlias(1:UpdateAliasRequest updateAliasRequest)throws (1:KmsException ex)

    /**
    * 删除已存在的别名所代表的主密钥（CMK）
    **/
    DeleteAliasResponse deleteAlias(1:DeleteAliasRequest deleteAliasRequest)throws (1:KmsException ex)

    /**
    * 查询当前用户所有的CMK
    **/
    ListAliasResponse listAlias(1:ListAliasRequest listAliasRequest)throws (1:KmsException ex)

    /**
    * 查询当前用户指定CMK信息
    **/
    ListAliasResponse listAliasesByKeyId(1:ListAliasesByKeyIdRequest listAliasesByKeyIdRequest)throws (1:KmsException ex)
    //////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                  用户CMK版本管理                                                   //
    //////////////////////////////////////////////////////////////////////////////////////////////////////
    /**
    *查询指定的密钥版本的元数据信息
    **/
    DescribeKeyVersionResponse describeKeyVersion(1: DescribeKeyVersionRequest describeKeyVersionRequest)throws (1:KmsException ex)
    /**
    *列出主密钥的所有密钥版本
    **/
    ListKeyVersionsResponse listKeyVersions(1:ListKeyVersionsRequest listKeyVersionsRequest)throws (1:KmsException ex)
    /**
    * 更新密钥轮转策略
    **/
    UpdateRotationPolicyResponse updateRotationPolicy(1:UpdateRotationPolicyRequest updateRotationPolicyRequest)throws (1:KmsException ex)
    /**
    *为主密钥创建一个新的密钥版本
    **/
    RotateKeyResponse rotateKey(1:RotateKeyRequest rotateKeyRequest)throws (1:KmsException ex)
  //////////////////////////////////////////////////////////////////////////////////////////////////////
    //                                 登录认证接口                                                 //
    //////////////////////////////////////////////////////////////////////////////////////////////////////
    TokenResponse getToken(1:TokenRequest tokenRequest)throws (1:KmsException ex)

    /**
    *创建租户加密的key
    **/
    CreateTenantKeyResponse createTenantKey(1:CreateTenantKeyRequest createTenantKeyRequest) throws (1:KmsException ex)

    /**
    *加密租户的secret Key
    **/
    EncryptSecretKeyResponse encryptSecretKey(1:EncryptSecretKeyRequest encryptSecretKeyRequest) throws (1:KmsException ex)

    /**
    *解密租户的secret Key
    **/
    DecryptSecretKeyResponse decryptSecretKey(1:DecryptSecretKeyRequest decryptSecretKeyRequest) throws (1:KmsException ex)

    /**
    * 在服务端使用datakey加密
    **/
    EncryptByDataKeyResponse encryptByDataKey(1:EncryptByDataKeyRequest encryptByDataKeyRequest) throws (1:KmsException ex)

    /**
    * 在服务端使用datakey解密
    **/
    DecryptByDataKeyResponse decryptByDataKey(1:DecryptByDataKeyRequest decryptByDataKeyRequest) throws (1:KmsException ex)
    /**
    * 密钥派生接口
    */
    DeriveKeyResponse deriveKey(DeriveKeyRequest request) throws (1:KmsException ex)
    /**
    * 导出对称密钥
    */
    ExportSymmetricKeyResponse exportSymmetricKey(ExportSymmetricKeyRequest request) throws (1:KmsException ex)

    HmacResponse hmac(HmacRequest request) throws (1:KmsException ex)

    HmacVerifyResponse hmacVerify(HmacVerifyRequest request) throws (1:KmsException ex)

    PingResponse ping(PingRequest request) throws (1:KmsException ex)

    KcvResponse getKcv(1:KcvRequest kcvRequest) throws (1:KmsException ex)
}
